import { loadTemplate } from "../helpers.js?ver=1";
import BookService from "../bookService/bookService.js?ver=1";
import AudioPlayerMenu from "../audioPlayerMenu/audioPlayerMenu.js?ver=1";
import Notification from "../notification/notification.js?ver=1";
import Auth from "../auth/auth.js?ver=1";

export default class AudioPlayer {
    constructor(url) {
        this.audioElement = null;
        this.currentFileIndex = 0;
        this.playlist = [];
		this.url = url;
		this.titleBook = '';
		this.imageBook = '';
		this.bookId = 0;
		this.service = null;
		this.playTime = 30000;
		this.timerId = 0;
		this.menu = null;
		this.notification = null;
		this.auth = null;
    }

    async init(callback) {
        try {
			this.notification = new Notification();
			this.service = new BookService(this.url, this.notification);
			this.auth = new Auth();
			this.auth.setPlayer(this);
			window.renderButton = () => this.auth.renderButton(window.openBook);
			
            this.content = await loadTemplate('audioPlayer');
            this.audio = this.content.cloneNode(true);
            document.body.append(this.audio);

			this.menu = new AudioPlayerMenu(this.service, this);
	
            this.initElements();
            this.initEventListeners();
            
            this.audioElement.volume = this.volumeControl.value;
            if (callback) callback();

			
        } catch (error) {
            console.error('Ошибка загрузки шаблона:', error);
        }
    }

	initData(email) {
		if(email)
		{
			this.service.setEmail(email);
		}
		
		this.audioElement.pause();
		
		this.service.getLastTime().then((lastTimeData) => {
			this.getPlayList(lastTimeData.bookid);
			this.resizePlayer();
			
			if(this.auth.isAuth()){
				this.refreshBtn.style.visibility = 'visible';
			}
			
			this.updatePlayPauseButton();
		});
	}
	
	getPlayList(bookId){
		this.service.getPlayList(bookId).then((playlistData) => {
			this.setPlaylist(playlistData);
		});
	}
	
	savePlayTime(start){
		
		if(!this.auth.isAuth()){
			return;
		}
		
		if((start && this.timerId > 0) || (!start && this.timerId == 0))
		{
			return;
		}
		
		if(start){
			this.timerId = setInterval(() => this.saveCurrentTrack(), this.playTime);
		}else{
			clearInterval(this.timerId);
			this.timerId = 0;
		}
	}

    initElements() {
        const elements = [
            'bookImageBox', 'refreshBtn', 'bookImage', 'playPauseBtn', 'nextBtn', 
            'prevBtn', 'seekBar', 'volumeControl', 'currentTimeDisplay', 'durationDisplay', 
            'audioElement', 'playlistContainer','titleBook'
        ];
        
        elements.forEach(el => this[el] = this.audio.querySelector(`#${el}`));
    }

    initEventListeners() {
        this.bookImage.addEventListener('click', () => this.togglePlayPause());
        this.refreshBtn.addEventListener('click', () => this.initData());
        this.playPauseBtn.addEventListener('click', () => this.togglePlayPause());
        this.nextBtn.addEventListener('click', () => this.nextTrack());
        this.prevBtn.addEventListener('click', () => this.previousTrack());
        this.seekBar.addEventListener('input', (e) => this.seekAudio(e));
        this.volumeControl.addEventListener('input', (e) => this.changeVolume(e));
        this.audioElement.addEventListener('timeupdate', () => this.updateSeekBar());
        this.audioElement.addEventListener('loadedmetadata', () => this.updateDuration());
        this.audioElement.addEventListener('ended', () => this.nextTrack());

        window.addEventListener('resize', () => this.resizePlayer());

	window.addEventListener('keydown', (event) => {
		if (event.key === 'P' || event.key === 'p') {
			this.togglePlayPause();
		}
	});

    }

    setPlaylist(playlistData) {
		
		const playlist = playlistData.list.map((obj, index) => ({
				id: index,
				title: obj.title,
				file: `${this.url}/${obj.mp3}`
		}));
		
		this.bookId = playlistData.lastPlay.bookid;
        this.playlist = playlist;
		this.titleBook.textContent = playlistData.lastPlay.name;
		this.bookImage.src = `${this.url}/${playlistData.lastPlay.img}`;
        this.setCurrentTrack(playlistData.lastPlay.filenum ?? 0, playlistData.lastPlay.time ?? 0.1);
		
		this.resizePlayer();
    }

    setCurrentTrack(fileIndex, time = 0.1) {
        this.currentFileIndex = fileIndex;
        this.setFile(this.playlist[fileIndex].file);
        if (time) this.audioElement.currentTime = time;
        this.renderPlaylist();
        this.highlightCurrentTrack();
    }

    getCurrentTrack() {
        return {
			bookId: this.bookId,
            file: this.currentFileIndex,
            time: this.audioElement.currentTime
        };
    }
	
	saveCurrentTrack(){
		
		if (this.audioElement.paused) {
			return false;
		}
		
		const info = this.getCurrentTrack();
		
		this.service.setPlayTime(info);
	}

    renderPlaylist() {
        this.playlistContainer.innerHTML = '';
        this.playlist.forEach((track, index) => {
            const trackElement = document.createElement('li');
            trackElement.className = 'track';
            trackElement.dataset.id = index;
            trackElement.textContent = track.title;
            trackElement.addEventListener('click', () => this.selectTrack(index));
            this.playlistContainer.appendChild(trackElement);
        });
    }

    highlightCurrentTrack() {
        [...this.playlistContainer.querySelectorAll('.track')].forEach(track => 
            track.classList.toggle('selected', track.dataset.id == this.currentFileIndex)
        );

        const currentTrack = this.playlistContainer.querySelector('.selected');
        currentTrack?.scrollIntoView({ behavior: 'smooth', block: 'center' });
    }

    selectTrack(index) {
        this.currentFileIndex = index;
        this.setFile(this.playlist[index].file);
		this.savePlayTime(true);
        this.audioElement.play();
        this.updatePlayPauseButton();
        this.highlightCurrentTrack();
		
    }

    nextTrack() {
        if (++this.currentFileIndex < this.playlist.length) {
            this.selectTrack(this.currentFileIndex);
        } else {
            console.log('Воспроизведение плейлиста завершено.');
        }
    }

    previousTrack() {
        if (--this.currentFileIndex >= 0) {
            this.selectTrack(this.currentFileIndex);
        } else {
            console.log('Нет предыдущего трека.');
        }
    }

    togglePlayPause() {
        if (this.audioElement.paused) {
			this.savePlayTime(true);
			this.audioElement.play();
        } else {
			this.savePlayTime(false);
            this.audioElement.pause();
        }
        this.updatePlayPauseButton();
    }

    updateSeekBar() {
        const currentTime = this.audioElement.currentTime;
        const duration = this.audioElement.duration;
        this.seekBar.value = (currentTime / duration) * 100;
        this.currentTimeDisplay.textContent = this.formatTime(currentTime);
    }

    seekAudio(event) {
        const seekTo = this.audioElement.duration * (event.target.value / 100);
        this.audioElement.currentTime = seekTo;
        this.audioElement.play();
        this.updatePlayPauseButton();
    }

    updateDuration() {
        this.durationDisplay.textContent = this.formatTime(this.audioElement.duration);
    }

    changeVolume(event) {
        this.audioElement.volume = event.target.value;
    }

    formatTime(seconds) {
        const minutes = Math.floor(seconds / 60);
        const remainingSeconds = Math.floor(seconds % 60);
        return `${minutes}:${remainingSeconds < 10 ? '0' : ''}${remainingSeconds}`;
    }

    setFile(src) {
        if (this.audioElement) {
            this.audioElement.src = src;
            this.audioElement.load();
			this.audioElement.currentTime = 0.0;
        } else {
            console.error('Аудиоэлемент не найден');
        }
    }

    updatePlayPauseButton() {
        const isPlaying = !this.audioElement.paused;
        this.playPauseBtn.classList.toggle('btn-play', !isPlaying);
        this.playPauseBtn.classList.toggle('btn-pause', isPlaying);
        this.bookImageBox.classList.toggle('play', !isPlaying);
    }

    resizePlayer() {
        const playerHeight = document.getElementById('player-box').offsetHeight;
        document.getElementById('play-list-box').style.height = `${window.innerHeight - playerHeight - 10}px`;
    }
}
